/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 � i-net software 1998-2013

*/
using System;
using System.Collections;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Text;
using System.Web;

namespace Inet.Viewer.Data
{
    /// <summary>
    /// The PageLoader interpretes the byte data of the jave protocotl and 
    /// delegates the rendering operations and information to the PagePaint.
    /// 
    /// The PageLoader processes the data for one page.
    /// </summary>
    public class PageLoader : Loader
    {
        private float alphaComposite = 1;
        /// <summary>
        /// painter for the rendering of the page
        /// </summary>        
        private readonly Graphics2DPainter painter;

        /// <summary>
        /// page information
        /// </summary>
        private readonly IPageReceiver receiver;

        /// <summary>
        /// Constructor </summary>
        /// <param name="painter"> Painter for the rendering </param>
        /// <param name="receiver"> PageView for page informationen </param>
        internal PageLoader(Graphics2DPainter painter, IPageReceiver receiver)
        {
            this.painter = painter;
            this.receiver = receiver;
        }

        /// <summary>
        /// Gets the painter-
        /// </summary>
        internal Graphics2DPainter Painter
        {
            get { return painter; }
        }

        internal IPageReceiver Receiver
        {
            get { return receiver; }
        }

        /// <summary>
        /// Makes sure the transmitted byte data are interpeted and forwarded 
        /// to the Painter and the Page.
        /// No connection to the server is done here, just the byte data for this page
        /// are parsed.
        /// </summary>
        /// <param name="pageData"> byte data that should be read</param>
        internal virtual void PaintPage(byte[] pageData)
        {
            painter.ClearElements();
            if (pageData != null)
            {
                this.Data = pageData;
                Render();
                painter.FinishPage();
            }
        }

        /// <summary>
        /// Falls die Exception eine NeedPrompts Exception ist, werden �ber
        /// die Methode "view.needPrompts" die Prompts abgefragt. Sollte
        /// der PromptDialog schon geöffnet sein, wird darauf gewartet, dass
        /// der PromptDialog fertig ist - dies soll daf�r sorgen, dass wenn
        /// z.B. mehrere Seiten eine NeedPrompts Exception gefeuert haben,
        /// nicht mehrere PromptDialoge ge�ffnet werden.
        /// </summary>        
        private void CheckPrompts(ViewerException exc)
        {
            if (exc.NeedPrompts)
            {
                throw new NotImplementedException();

                /*if (view.PromptDialogOpen)
                {
                    // diese Schelif �berarbeiten
                 * while (view.PromptDialogOpen)
                    {      
                        Monitor.Wait(this); // oder Monitor.Wait(view);
                        //Thread.yield();
                    }
                }
                {
                    view.RequestPrompts(exc.Prompts);
                }*/
            }
        }

        /// <summary>
        /// To initialize the renderer
        /// </summary>
        private void Render()
        {
            // set default colors of the renderer as they are not transmitted            
            painter.Brush = new SolidBrush(Color.FromArgb(0xFFFFFF)); // transparent white
            painter.Pen = new Pen(Color.Black);

            ReadTokens();
        }

        /// <summary>
        /// <inherit/>
        /// </summary>
        /// <seealso cref= "Inet.Viewer.Data.Loader.Load(int)"/>
        protected internal override bool Load(int token)
        {
            switch (token)
            {
                case ViewerTokenConstants.TokenFonts: // Fonts  
                    while (Offset < OldOffset + TokenSize)
                    {
                        ReadInt(); // (TOKEN_FONT)
                        int tmpTokenSize = ReadInt(); // TOKEN_FONT SIZE
                        int tmpOffset = Offset;
                        string fontName = ReadString();
                        int fontStyle = ReadInt();
                        int fontSize = ReadInt();
                        int fontID = ReadInt();
                        int fontRev = ReadInt();
                        if (fontID > 0 && fontRev > 0)
                        {
                            // is embedded font and has to be fetched                            
                            // Embedded Fonts werden zentral im ReportScrollView gespeichert.
                            // Darauf greifen wir indirekt �ber die PageView zu.                        
                            Font embeddedFont = receiver.GetEmbeddedFont(fontID, fontRev);
                            if (embeddedFont.Style == (FontStyle)fontStyle)
                            {
                                painter.AddFont(embeddedFont, fontStyle, fontSize);
                            }
                            else
                            {
                                // Exception for fonts with fonts that include the style
                                // in the font name and don't support regular: "Eg.Arial Bold"
                                // get next available fontstyle
                                FontStyle firstStyle = FontLoader.GetAvailableStyle(embeddedFont.FontFamily) | (FontStyle)(fontStyle & 0xC);
                                painter.AddFont(embeddedFont, (int)firstStyle, fontSize);
                            }
                        }
                        else
                        {
                            painter.AddFont(fontName, fontStyle, fontSize);
                        }
                        Offset = tmpOffset + tmpTokenSize;
                    }
                    break;
                case ViewerTokenConstants.TokenAdornments: // Adornment
                    while (Offset < OldOffset + TokenSize)
                    {
                        ReadInt(); // (TOKEN_ADORNMENT)
                        int tmpTokenSize = ReadInt(); // TOKEN_ADORNMENT SIZE
                        int tmpOffset = Offset;
                        painter.AddAdornment(new Adornment((LineStyle)ReadInt(), (LineStyle)ReadInt(), (LineStyle)ReadInt(), (LineStyle)ReadInt(), ReadInt(), ReadInt(), ReadInt(), ReadInt() != 0));
                        Offset = tmpOffset + tmpTokenSize;
                    }
                    break;
                case ViewerTokenConstants.TokenSetCurrentFont:
                    {
                        int id = ReadInt();
                        painter.Font = id;
                    }
                    break;
                case ViewerTokenConstants.TokenSetCurrentDrawColor:
                    Color drawColor = Color.FromArgb(Read4ByteInt());
                    // In C# there is no AlphaComposite class:
                    if (alphaComposite < 1)
                    {
                        drawColor = Color.FromArgb((int)(alphaComposite * 256), drawColor);
                    }
                    painter.Pen.Color = drawColor;
                    break;
                case ViewerTokenConstants.TokenSetCurrentFillColor:
                    Color fillColor = Color.FromArgb(Read4ByteInt());
                    // In C# there is no AlphaComposite class:
                    if (alphaComposite < 1)
                    {
                        fillColor = Color.FromArgb((int)(alphaComposite * 256), fillColor);
                    }
                    painter.Brush = new SolidBrush(fillColor);
                    break;
                case ViewerTokenConstants.TokenSetCurrentDrawPaint:
                    LinearGradientBrush gradBrush = ReadGradientPaint(alphaComposite);
                    painter.Brush = gradBrush;

                    break;
                case ViewerTokenConstants.TokenSetCurrentFillPaint:
                    painter.Brush = ReadGradientPaint(alphaComposite);
                    break;
                case ViewerTokenConstants.TokenSetCurrentTextureDrawPaint:
                case ViewerTokenConstants.TokenSetCurrentTextureFillPaint:
                    painter.Brush = ReadTexturePaint(alphaComposite);
                    break;
                case ViewerTokenConstants.TokenSetCurrentAdornment:
                    {
                        int id = ReadInt();
                        painter.Adornment = id;
                    }
                    break;
                case ViewerTokenConstants.TokenReportInfo: // Report Info ALTER viewer: page-info was collected in Load() ?
                    {
                        ReportInfo info = new ReportInfo();

                        int flags = ReadInt();
                        info.IsFormPrint = (flags & 1) > 0;
                        info.IsPrintingEnabled = (flags & 2) > 0;
                        info.IsClipboardEnabled = (flags & 4) > 0;
                        info.IsExportEnabled = (flags & 8) > 0;
                        info.IsGroupTreeVisible = (flags & 16) > 0;
                        info.IsFontScaling = (flags & 32) > 0;
                        info.IsUserPageFormat = (flags & 64) > 0;
                        info.IsStrongPDFEncryption = (flags & 128) > 0;
                        info.IsReportSuppressed = (flags & 256) > 0;

                        // allowed export formats
                        int count = ReadInt();
                        info.Formats = new HashSet<string>();
                        for (int i = 0; i < count; i++)
                        {
                            info.Formats.Add(ReadString());
                        }

                        info.Title = ReadString();
                        info.FileName = ReadString();

                        if (!receiver.WriteReportInfo(info, this))
                        {
                            return false;
                        }
                        if (painter != null)
                        {
                            painter.WriteReportInfo(info);
                        }
                        break;
                    }
                case ViewerTokenConstants.TokenPageInfo:
                    {
                        PageInfo info = new PageInfo();
                        info.PageNr = ReadInt();
                        info.PageWidth = ReadInt();
                        info.PageHeight = ReadInt();
                        info.BorderLeft = ReadInt();
                        info.BorderRight = ReadInt();
                        info.BorderTop = ReadInt();
                        info.BorderBottom = ReadInt();
                        info.PageAlign = ReadInt();
                        info.Timestamp = Read8ByteLong();
                        if (!receiver.WritePageInfo(info, this))
                        {
                            return false;
                        }
                        painter.WritePageInfo(info);
                        break;
                    }
                case ViewerTokenConstants.TokenClip: // Clip

                    int x = ReadTwip();
                    int y = ReadTwip();
                    int width = ReadTwip();
                    int height = ReadTwip();
                    string toolTip = ReadString();
                    if (toolTip != null)
                    {
                        painter.AddHyperlink(new PageClip(x, y, width, height, toolTip));
                    }
                    painter.SetClip(x, y, width, height);
                    break;
                case ViewerTokenConstants.TokenResetClip: // Reset Clip
                    painter.ResetClip();
                    break;
                case ViewerTokenConstants.TokenDrawString: // Draw String
                    x = ReadTwip();
                    y = ReadTwip();
                    int align = ReadInt();
                    int rotGrad = ReadInt();
                    /*
                     * value= (textrotation | glyph-direction) Textrotation: 0�, 90�, 180�, or 270� glyph direction: 512 * DIRECTION (0 for UP, 1 for RIGHT, 2 for DOWN, 3 f�r LEFT)
                     */
                    int glyphOrientation = rotGrad / 512;
                    rotGrad = rotGrad % 512;
                    width = ReadTwip();
                    int maxWidth = ReadTwip();

                    string str = ReadString();
                    if (str == null || str.Length == 0)
                    {
                        break;
                    }
                    painter.DrawString(str, x, y, rotGrad, align, glyphOrientation, width, maxWidth);
                    break;
                case ViewerTokenConstants.TokenDrawStringG2D: // Draw String
                     float fx = ReadFloat();
                    float fy = ReadFloat();
                    float fwidth = ReadFloat();
                    str = ReadString();
                    painter.DrawString( str, fx, fy, fwidth);
                    break;
                case ViewerTokenConstants.TokenDrawLine: // Draw Line
                    int x1 = ReadTwip();
                    int y1 = ReadTwip();
                    int x2 = ReadTwip();
                    int y2 = ReadTwip();

                    painter.DrawLine(x1, y1, x2, y2);
                    break;
                case ViewerTokenConstants.TokenDrawLineElement:
                    painter.DrawLine(ReadTwip(), ReadTwip(), ReadTwip(), ReadTwip(), (LineStyle)ReadInt(), ReadTwip()); // line width -  style -  y2 -  x2 -  y1 -  x1
                    break;
                case ViewerTokenConstants.TokenDrawRect: // Draw Rect
                    x = ReadTwip();
                    y = ReadTwip();
                    width = ReadTwip();
                    height = ReadTwip();
                    painter.DrawRect(x, y, width, height);
                    break;
                case ViewerTokenConstants.TokenDrawFillRect: // Draw Fill Rect
                    x = ReadTwip();
                    y = ReadTwip();
                    width = ReadTwip();
                    height = ReadTwip();
                    painter.DrawFillRect(x, y, width, height);
                    break;
                case ViewerTokenConstants.TokenFillRectSmooth: // Fill Rect
                    x = ReadTwip();
                    y = ReadTwip();
                    width = ReadTwip();
                    height = ReadTwip();
                    painter.FillRect(x, y, width, height, true);
                    break;
                case ViewerTokenConstants.TokenFillRect: // Fill Rect
                    x = ReadTwip();
                    y = ReadTwip();
                    width = ReadTwip();
                    height = ReadTwip();
                    painter.FillRect(x, y, width, height);
                    break;
                case ViewerTokenConstants.TokenDrawRoundRect:
                    painter.DrawRoundRect(ReadTwip(), ReadTwip(), ReadTwip(), ReadTwip(), ReadTwip(), ReadTwip());
                    break;
                case ViewerTokenConstants.TokenFillRoundRect:
                    painter.FillRoundRect(ReadTwip(), ReadTwip(), ReadTwip(), ReadTwip(), ReadTwip(), ReadTwip());
                    break;
                case ViewerTokenConstants.TokenDrawImage: // Draw Image
                    x = ReadTwip();
                    y = ReadTwip();
                    width = ReadTwip();
                    height = ReadTwip();
                    int leftInset = ReadTwip();
                    int rightInset = ReadTwip();
                    int topInset = ReadTwip();
                    int bottomInset = ReadTwip();
                    // not used as not implemented on server side
                    int scalingHint = ReadTwip();

                    byte[] imgBytes = ReadByteArray();
                    if (imgBytes != null)
                    {
                        // "Hash"-value currently not needed
                        painter.DrawImage(imgBytes, x + leftInset, y + rightInset, width - leftInset - rightInset, height - bottomInset - topInset, 0);
                    }

                    break;
                case ViewerTokenConstants.TokenDrawPolygon: // Draw Polygon
                    painter.DrawPolygon(ReadPolygon());
                    break;
                case ViewerTokenConstants.TokenFillPolygon: // Fill Polygon
                    painter.FillPolygon(ReadPolygon());
                    break;
                case ViewerTokenConstants.TokenDrawFillPolygon: // Draw and Fill Polygon
                    painter.DrawPolygon(ReadPolygon());
                    painter.FillPolygon(ReadPolygon());
                    break;
                case ViewerTokenConstants.TokenDrawOval: // Draw Oval
                    x = ReadTwip();
                    y = ReadTwip();
                    width = ReadTwip();
                    height = ReadTwip();

                    painter.DrawOval(x, y, width, height);
                    break;
                case ViewerTokenConstants.TokenFillOval: // FillOval
                    x = ReadTwip();
                    y = ReadTwip();
                    width = ReadTwip();
                    height = ReadTwip();

                    painter.FillOval(x, y, width, height);
                    break;
                case ViewerTokenConstants.TokenDrawArc: // Draw Arc
                    x = ReadTwip();
                    y = ReadTwip();
                    width = ReadTwip();
                    height = ReadTwip();
                    x1 = ReadInt();
                    x2 = ReadInt();

                    painter.DrawArc(x, y, width, height, x1, x2);
                    break;
                case ViewerTokenConstants.TokenFillArc: // Fill Arc
                    x = ReadTwip();
                    y = ReadTwip();
                    width = ReadTwip();
                    height = ReadTwip();
                    x1 = ReadInt();
                    x2 = ReadInt();
                    painter.FillArc(x, y, width, height, x1, x2);
                    break;
                case ViewerTokenConstants.TokenCreate:
                    painter.Create(ReadInt());
                    break;
                case ViewerTokenConstants.TokenDispose:
                    painter.Dispose();
                    break;
                case ViewerTokenConstants.TokenChangeGraphics:
                    painter.ChangeCurrentGraphics(ReadInt());
                    break;
                case ViewerTokenConstants.TokenTransform:
                    Matrix transf = ReadTransform();
                    painter.Transform(transf);
                    break;
                case ViewerTokenConstants.TokenSetTransform:
                    Matrix transform = ReadTransform();
                    painter.SetTransform(transform);
                    break;
                case ViewerTokenConstants.TokenSetClipShape:
                    GraphicsPath path = ReadShape();
                    painter.SetClip(path);
                    break;
                case ViewerTokenConstants.TokenDrawShape:
                    painter.Draw(ReadShape());
                    break;
                case ViewerTokenConstants.TokenFillShape:
                    painter.Fill(ReadShape());
                    break;
                case ViewerTokenConstants.TokenSetStroke:
                    float sWidth = ReadFloat();
                    int cap = ReadInt();
                    int join = ReadInt();
                    float miterlimit = ReadFloat();
                    float[] dash = ReadFloatArray();

                    if (dash != null)
                    {
                        // the dash in C# is the product of the dash and the width of the Pen. But the dashes we receive are absolut.                        
                        for (int i = 0; i < dash.Length; i++)
                        {
                            dash[i] = dash[i] / sWidth;
                        }
                        painter.Pen.DashPattern = dash;
                        painter.Pen.DashStyle = DashStyle.Custom;
                    }
                    else
                    {
                        painter.Pen.DashStyle = DashStyle.Solid;
                    }
                    float dash_phase = ReadFloat();
                    painter.Pen.Width = sWidth;
                    painter.Pen.MiterLimit = miterlimit;
                    painter.Pen.DashOffset = dash_phase;

                    // set line join
                    LineJoin lineJoin = LineJoin.Miter;
                    switch (join)
                    {
                        case 0:
                            lineJoin = LineJoin.Miter;
                            break;
                        case 1:
                            lineJoin = LineJoin.Round;
                            break;
                        case 2:
                            lineJoin = LineJoin.Bevel;
                            break;
                    }
                    painter.Pen.LineJoin = lineJoin;

                    // Set DashCap
                    DashCap dashCap = DashCap.Flat;
                    switch (join)
                    {
                        case 0:
                            dashCap = DashCap.Flat;
                            break;
                        case 1:
                            dashCap = DashCap.Round;
                            break;
                        case 2:
                            dashCap = DashCap.Flat;
                            break;
                    }
                    painter.Pen.DashCap = dashCap;
                    break;
                case ViewerTokenConstants.TokenSetComposite:
                    // rule not used in C#
                    int rule = ReadInt();
                    this.alphaComposite = ReadFloat();
                    break;
                case ViewerTokenConstants.TokenHyperlink: // HyperLink
                    x = ReadInt();
                    y = ReadInt();
                    width = ReadInt();
                    height = ReadInt();
                    LinkType type = (LinkType)ReadInt();
                    string urlString = ReadString();
                    string str2 = ReadString();

                    if (type == LinkType.SubreportOnDemand || type == LinkType.InteractiveSorting)
                    {
                        try
                        {
                            Encoding codePage = Encoding.UTF8;
                            urlString = HttpUtility.UrlDecode(urlString);
                        }
                        catch (Exception e)
                        {
                            // Than it will not be coded,  but should never occur
                            ViewerUtils.Error(e.ToString());
                        }
                        painter.AddSubreportOnDemand(new PageClip(x, y, width, height, type, urlString, str2));
                    }
                    else if (type == LinkType.Hyperlink)
                    {
                        Uri url = null;
                        try
                        {
                            url = new Uri(urlString);
                        }
                        catch (ArgumentNullException)
                        {
                            break;
                        }
                        catch (UriFormatException)
                        {
                            break;
                        }
                        painter.AddHyperlink(new PageClip(x, y, width, height, type, url,str2));
                    }
                    break;
                default:
                    return base.Load(token); // case 1: Error message, usw.
            }
            return true;
        }
    }
}